/*
	Example applet of Conway's Life.
	1996 Peter N Lewis <peter@stairways.com.au>
	Written for Metrowerks
	
	Feel free to use this source code in any way.
*/

package au.com.peter.example;

import java.awt.*;
import java.applet.Applet;


public class Life extends Applet
{
	public void init()
	{
		Board p;
		Checkbox c;
		
		setLayout( new BorderLayout() );
		
		c = new Checkbox( "Running" );
		c.setState( true );
		add( "North", c );
		p = new Board( 10, 10, 100, 100, c);
		add( "Center", p );

		resize(300, 300);
		//p.repaint();
		p.start();
	}  // end init()

	public static void main(String args[])
	{
		com.metrowerks.AppletFrame.startApplet("au.com.peter.example.Life", "Life", args);
	}

}


class Board extends Canvas implements Runnable {
	
	int x_max, y_max;
	Checkbox running;
	boolean board[][];
	private boolean temp_board[][];
	
	public Board( int x_max, int y_max, int x_size, int y_size, Checkbox running ) {
		this.x_max = x_max;
		this.y_max = y_max;
		this.board = new boolean[x_max][y_max];
		this.temp_board = new boolean[x_max][y_max];
		this.running = running;
		
		this.board[2][2] = true;
		this.board[2][3] = true;
		this.board[2][4] = true;
		this.board[1][4] = true;
		this.board[0][3] = true;
		
		resize( x_size, y_size );
		repaint();
	}
	
	private int XtoCell( int x ) {
		int x_total = bounds().width+1;
		
		return x * x_max / x_total;
	}
	
	private int YtoCell( int y ) {
		int y_total = bounds().height+1;
		
		return y * y_max / y_total;
	}
	
	private void PaintCell( Graphics g, Rectangle b, int x, int y ) {
		int x_total = b.width -1;
		int y_total = b.height -1;
		int left, right, top, bottom, width, height;
		
		left = x * x_total / x_max;
		right = (x+1) * x_total / x_max;
		width = right - left;
		top = y * y_total / y_max;
		bottom = (y+1) * y_total / y_max;
		height = bottom - top;
		
		g.setColor(Color.black);
		if ( board[x][y] ) {
			g.fillRect(left, top, width, height);
		} else {
			g.drawRect(left, top, width, height);
			g.clearRect(left+1, top+1, width-1, height-1);
		}
	}
	
	private void PaintCell( int x, int y ) {
		Graphics g = getGraphics();
		
		if ( g != null ) {
			PaintCell( g, bounds(), x, y );
		}
	}
	
	public void paint( Graphics g ) {
		Rectangle b = bounds();

		for ( int y = 0; y < y_max; y++ ) {
			for ( int x = 0; x < x_max; x++ ) {
				PaintCell( g, b, x, y );
			}
		}
	}
	
	boolean dragging = false;
	
	public synchronized void Propagate() {
		
		if ( dragging ) {
			return;
		}
		
		for ( int y = 0; y < y_max; y++ ) {
			for ( int x = 0; x < x_max; x++ ) {
				int sum = 0;
				
				for ( int dy = -1; dy <= 1; dy++ ) {
					for ( int dx = -1; dx <= 1; dx++ ) {
						if ( dx != 0 || dy != 0 ) {
							int tx = x + dx;
							int ty = y + dy;
							
							if ( tx < 0 ) {
								tx = x_max - 1;
							}
							if ( tx >= x_max ) {
								tx = 0;
							}
							if ( ty < 0 ) {
								ty = y_max - 1;
							}
							if ( ty >= y_max ) {
								ty = 0;
							}
							if ( board[tx][ty] ) {
								sum++;
							}
						}
					}
				}
				
				if ( board[x][y] && ((sum<2) || (sum>=4)) ) {
					temp_board[x][y] = false;
				} else if ( !board[x][y] && sum==3 ) {
					temp_board[x][y] = true;
				} else {
					temp_board[x][y] = board[x][y];
				}
			}
		}

		Graphics g = getGraphics();
		Rectangle b = bounds();

		if ( g != null ) {
			for ( int y = 0; y < y_max; y++ ) {
				for ( int x = 0; x < x_max; x++ ) {
					if ( temp_board[x][y] != board[x][y] ) {
						board[x][y] = temp_board[x][y];
						PaintCell( g, b, x, y );
					}
				}
			}
		}
	}
	
	int drag_x;
	int drag_y;
	boolean inside;
	
	private void CheckInside( int x, int y ) {
		int tmp_x = XtoCell( x );
		int tmp_y = YtoCell( y );
		boolean tmp_inside = (tmp_x == drag_x) && (tmp_y == drag_y);
		
		if ( tmp_inside != inside ) {
			inside = tmp_inside;
			board[drag_x][drag_y] = !board[drag_x][drag_y];
			PaintCell( drag_x, drag_y );
		}
	}
	
	public boolean mouseDown( Event e, int x, int y ) {
		drag_x = XtoCell( x );
		drag_y = YtoCell( y );
		dragging = (0 <= drag_x) && (drag_x < x_max) && (0 <= drag_y) && (drag_y < y_max);
		if ( dragging ) {
			synchronized ( this ) {
				inside = true;
				board[drag_x][drag_y] = !board[drag_x][drag_y];
				PaintCell( drag_x, drag_y );
			}
		}
		return true;
	}
	
	public boolean mouseUp( Event e, int x, int y ) {
		if ( dragging ) {
			CheckInside( x, y );
			dragging = false;
		}
		return true;
	}
	
	public boolean mouseDrag( Event e, int x, int y ) {
		if ( dragging ) {
			CheckInside( x, y );
		}
		return true;
	}
	
	Thread thread = null;
	
	public void start() {
		if ( thread == null ) {
			thread = new Thread( this );
			thread.start();
		}
	}
	
	public void stop() {
		if ( thread != null && thread.isAlive() ) {
			thread.stop();
		}
		thread = null;
	}
	
	public void run() {
		while (true) {
			try { Thread.sleep(25); } catch (InterruptedException e){};
			if ( isVisible() && running.getState() ) {
				Propagate();
			}
		}
	}
	
}

